<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Properties_Google_Map_Widget extends \Elementor\Widget_Base {

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );
	}

	public function get_name() {
		return 'rhea-properties-google-map-widget';
	}

	public function get_title() {
		return esc_html__( 'RH: Properties Google Map', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		// More classes for icons can be found at https://pojome.github.io/elementor-icons/
		return 'eicon-google-maps rh-modern-widget';
	}

	public function get_categories() {
		return [ 'real-homes' ];
	}

	public function rhea_google_properties_script() {

		$google_map_arguments = array();

		wp_enqueue_script(
			'google-map-api',
			esc_url_raw(
				add_query_arg(
					apply_filters(
						'inspiry_google_map_arguments',
						$google_map_arguments
					),
					'//maps.google.com/maps/api/js'
				)
			),
			array(),
			false,
			false
		);

		wp_enqueue_script(
			'google-map-marker-clusterer',
			RHEA_PLUGIN_URL . 'elementor/js/markerclusterer.js',
			array( 'google-map-api' ),
			'2.1.1'
		);

		wp_enqueue_script(
			'google-map-oms',
			RHEA_PLUGIN_URL . 'elementor/js/oms.min.js',
			array( 'google-map-api' ),
			'0.3.3'
		);

		wp_enqueue_script(
			'google-map-info-box',
			RHEA_PLUGIN_URL . 'elementor/js/infobox.js',
			array( 'google-map-api' ),
			'1.1.9'
		);

		wp_enqueue_script(
			'rhea-properties-google-map-js',
			RHEA_PLUGIN_URL . 'elementor/js/google-map.js',
			[ 'google-map-api', 'elementor-frontend' ],
			RHEA_VERSION,
			true
		);

	}

	protected function register_controls() {

		$allowed_html = array(
			'a'      => array(
				'href'  => array(),
				'title' => array()
			),
			'br'     => array(),
			'em'     => array(),
			'strong' => array(),
		);

		$this->start_controls_section(
			'ere_properties_section',
			[
				'label' => esc_html__( 'Properties', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'map_sync_properties',
			[
				'label'              => esc_html__( 'Sync Properties Widget Data', RHEA_TEXT_DOMAIN ),
				'description'        => esc_html__( 'Use this map to sync properties from Ultra Properties Widget.', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true,
			]
		);

		$this->add_control(
			'all_found_properties',
			[
				'label'              => esc_html__( 'Show All Found Properties', RHEA_TEXT_DOMAIN ),
				'description'        => esc_html__( 'If enabled, all available properties will be displayed (May impact performance with large number of properties).', RHEA_TEXT_DOMAIN ),
				'type'               => \Elementor\Controls_Manager::SWITCHER,
				'label_on'           => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'          => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value'       => 'yes',
				'default'            => 'no',
				'frontend_available' => true
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'     => esc_html__( 'Number of Properties', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'min'       => 1,
				'max'       => 1000,
				'step'      => 1,
				'default'   => 6,
				'condition' => [
					'map_sync_properties'  => [ 'no', '' ],
					'all_found_properties' => [ 'no', '' ]
				]
			]
		);

		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$options = [];
				$terms   = get_terms( $single_tax->name );

				if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
					foreach ( $terms as $term ) {
						$options[ $term->slug ] = $term->name;
					}
				}

				$this->add_control(
					$single_tax->name,
					[
						'label'       => $single_tax->label,
						'type'        => \Elementor\Controls_Manager::SELECT2,
						'multiple'    => true,
						'label_block' => true,
						'options'     => $options,
						'condition'   => [
							'map_sync_properties' => [ 'no', '' ]
						],
					]
				);
			}
		}

		$this->add_control(
			'orderby',
			[
				'label'     => esc_html__( 'Order By', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'date'       => esc_html__( 'Date', RHEA_TEXT_DOMAIN ),
					'price'      => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
					'title'      => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
					'menu_order' => esc_html__( 'Menu Order', RHEA_TEXT_DOMAIN ),
					'rand'       => esc_html__( 'Random', RHEA_TEXT_DOMAIN ),
				],
				'default'   => 'date',
				'condition' => [
					'map_sync_properties' => [ 'no', '' ]
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'     => esc_html__( 'Order', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'asc'  => esc_html__( 'Ascending', RHEA_TEXT_DOMAIN ),
					'desc' => esc_html__( 'Descending', RHEA_TEXT_DOMAIN ),
				],
				'default'   => 'desc',
				'condition' => [
					'map_sync_properties' => [ 'no', '' ]
				],
			]
		);

		$this->add_control(
			'show_only_featured',
			[
				'label'        => esc_html__( 'Show Only Featured Properties', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'map_sync_properties' => [ 'no', '' ]
				],
			]
		);

		$this->add_control(
			'skip_sticky_properties',
			[
				'label'        => esc_html__( 'Skip Sticky Properties', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'condition'    => [
					'map_sync_properties' => [ 'no', '' ]
				],
			]
		);

		$this->add_control(
			'offset',
			[
				'label'     => esc_html__( 'Offset or Skip From Start', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::NUMBER,
				'default'   => '0',
				'condition' => [
					'map_sync_properties' => [ 'no', '' ]
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'ere_google_map_options',
			[
				'label' => esc_html__( 'Google Map Options', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'map_styles',
			[
				'label'   => esc_html__( 'Map Styles', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'roadmap'   => esc_html__( 'RoadMap', RHEA_TEXT_DOMAIN ),
					'satellite' => esc_html__( 'Satellite', RHEA_TEXT_DOMAIN ),
					'hybrid'    => esc_html__( 'Hybrid', RHEA_TEXT_DOMAIN ),
					'terrain'   => esc_html__( 'Terrain', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'roadmap',
			]
		);


		$this->add_control(
			'cta_map_style_json',
			[
				'label'       => esc_html__( 'Google Maps Styles JSON (optional)', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::TEXTAREA,
				'rows'        => 10,
				'description' => wp_kses( __( 'You can create Google Maps styles JSON using <a href="https://mapstyle.withgoogle.com" target="_blank">Google Styling Wizard</a> or <a href="https://snazzymaps.com/" target="_blank">Snazzy Maps</a>', RHEA_TEXT_DOMAIN ), $allowed_html ),

			]
		);


		$this->end_controls_section();
		$this->start_controls_section(
			'ere_property_typo_section',
			[
				'label' => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_heading_typography',
				'label'    => esc_html__( 'Popup Title', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .prop-title a',
			]
		);


		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_typography',
				'label'    => esc_html__( 'Popup Price', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea-popup-price, {{WRAPPER}} .map-info-window .rh-gm-thumb-detail p',
			]
		);


		$this->end_controls_section();

		$this->start_controls_section(
			'ere_property_style_section',
			[
				'label' => esc_html__( 'Spacings And Sizes', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);


		$this->add_responsive_control(
			'ere_property_map_height',
			[
				'label'           => esc_html__( 'Map Height (px)', RHEA_TEXT_DOMAIN ),
				'type'            => \Elementor\Controls_Manager::SLIDER,
				'range'           => [
					'px' => [
						'min' => 0,
						'max' => 1000,
					],
				],
				'desktop_default' => [
					'size' => 600,
					'unit' => 'px',
				],
				'selectors'       => [
					'{{WRAPPER}} .rhea-listing-google-map' => 'height: {{SIZE}}{{UNIT}};',

				],
			]
		);

		$this->add_responsive_control(
			'ere_property_popup_title_Padding',
			[
				'label'      => esc_html__( 'Popup Title Padding (px)', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .prop-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'ere_property_popup_price_Padding',
			[
				'label'      => esc_html__( 'Popup Price Padding (px)', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rhea-popup-price, {{WRAPPER}} .map-info-window .rh-gm-thumb-detail p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		if ( "ultra" !== INSPIRY_DESIGN_VARIATION ) {
			$this->add_responsive_control(
				'ere_property_popup_border_size',
				[
					'label'     => esc_html__( 'Popup Border Bottom Size', RHEA_TEXT_DOMAIN ),
					'type'      => \Elementor\Controls_Manager::SLIDER,
					'range'     => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .rhea-map-info-window' => 'border-bottom-width: {{SIZE}}{{UNIT}};',
					],
				]
			);
		}

		$this->end_controls_section();

		$this->start_controls_section(
			'ere_property_colors_section',
			[
				'label' => esc_html__( 'Colors', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'rhea_popup_content_bg_color',
			[
				'label'     => esc_html__( 'Popup Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rhea-map-info-window, {{WRAPPER}} .map-info-window' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_popup_title_color',
			[
				'label'     => esc_html__( 'Popup Title Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .prop-title a' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'rhea_popup_title_hover_color',
			[
				'label'     => esc_html__( 'Popup Title Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .prop-title a:hover' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'rhea_popup_price_color',
			[
				'label'     => esc_html__( 'Popup Price Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rhea-popup-price, {{WRAPPER}} .map-info-window .rh-gm-thumb-detail p' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_popup_wrapper_border_color',
			[
				'label'     => esc_html__( 'Popup Border and Tip Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'default'   => '',
				'selectors' => [
					'{{WRAPPER}} .rhea-map-info-window' => 'border-bottom-color: {{VALUE}}',
					'{{WRAPPER}} .arrow-down'           => 'border-top-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();
	}

	public function inject_content() {
		$settings = $this->get_settings_for_display();

		// Remove sticky properties filter.
		if ( $settings['skip_sticky_properties'] == 'yes' ) {
			remove_filter( 'the_posts', 'inspiry_make_properties_stick_at_top', 10 );
		}

		if ( get_query_var( 'paged' ) ) {
			$paged = get_query_var( 'paged' );
		} else if ( get_query_var( 'page' ) ) { // if is static front page
			$paged = get_query_var( 'page' );
		} else {
			$paged = 1;
		}

		if ( $settings['offset'] ) {
			$offset = $settings['offset'] + ( $paged - 1 ) * $settings['posts_per_page'];
		} else {
			$offset = '';
		}

		// Basic Query
		$properties_args = array(
			'post_type'      => 'property',
			'posts_per_page' => ( $settings['all_found_properties'] === 'yes' ) ? -1 : $settings['posts_per_page'],
			'order'          => $settings['order'],
			'offset'         => $offset,
			'post_status'    => 'publish',
			'paged'          => $paged,
		);

		// Sorting
		if ( 'price' === $settings['orderby'] ) {
			$properties_args['orderby']  = 'meta_value_num';
			$properties_args['meta_key'] = 'REAL_HOMES_property_price';
		} else {
			// for date, title, menu_order and rand
			$properties_args['orderby'] = $settings['orderby'];
		}

		// Filter based on custom taxonomies
		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$setting_key = $single_tax->name;
				if ( ! empty( $settings[ $setting_key ] ) ) {
					$properties_args['tax_query'][] = [
						'taxonomy' => $setting_key,
						'field'    => 'slug',
						'terms'    => $settings[ $setting_key ],
					];
				}
			}

			if ( isset( $properties_args['tax_query'] ) && count( $properties_args['tax_query'] ) > 1 ) {
				$properties_args['tax_query']['relation'] = 'AND';
			}
		}

		$meta_query = array();
		if ( 'yes' === $settings['show_only_featured'] ) {
			$meta_query[] = array(
				'key'     => 'REAL_HOMES_featured',
				'value'   => 1,
				'compare' => '=',
				'type'    => 'NUMERIC',
			);

			$properties_args['meta_query'] = $meta_query;
		}

		$map_properties_query = new WP_Query( apply_filters( 'rhea_google_maps_properties_widget', $properties_args ) );
		$properties_map_data  = array();

		if ( $map_properties_query->have_posts() ) {
			while ( $map_properties_query->have_posts() ) {
				$map_properties_query->the_post();

				$property_id = get_the_ID();

				$current_property_data          = array();
				$current_property_data['title'] = get_the_title();

				$current_property_data['address'] = get_post_meta( $property_id, 'REAL_HOMES_property_address', true );
				$current_property_data['type']    = ere_get_property_types( $property_id );

				if ( function_exists( 'ere_get_property_price' ) ) {
					$current_property_data['price'] = ere_get_property_price();
				} else {
					$current_property_data['price'] = null;
				}

				$current_property_data['url'] = get_permalink();

				// property location
				$property_location = get_post_meta( $property_id, 'REAL_HOMES_property_location', true );
				if ( ! empty( $property_location ) ) {
					$lat_lng                      = explode( ',', $property_location );
					$current_property_data['lat'] = $lat_lng[0];
					$current_property_data['lng'] = $lat_lng[1];
				}

				// property thumbnail
				if ( has_post_thumbnail() ) {
					$image_id         = get_post_thumbnail_id();
					$image_attributes = wp_get_attachment_image_src( $image_id, 'property-thumb-image' );
					if ( ! empty( $image_attributes[0] ) ) {
						$current_property_data['thumb'] = $image_attributes[0];
					}
				}

				// Property map icon based on Property Type
				$type_terms = get_the_terms( $property_id, 'property-type' );
				if ( $type_terms && ! is_wp_error( $type_terms ) ) {
					foreach ( $type_terms as $type_term ) {
						$icon_id = get_term_meta( $type_term->term_id, 'inspiry_property_type_icon', true );
						if ( ! empty ( $icon_id ) ) {
							$icon_url = wp_get_attachment_url( $icon_id );
							if ( $icon_url ) {
								$current_property_data['icon'] = esc_url( $icon_url );

								// Retina icon
								$retina_icon_id = get_term_meta( $type_term->term_id, 'inspiry_property_type_icon_retina', true );
								if ( ! empty ( $retina_icon_id ) ) {
									$retina_icon_url = wp_get_attachment_url( $retina_icon_id );
									if ( $retina_icon_url ) {
										$current_property_data['retinaIcon'] = esc_url( $retina_icon_url );
									}
								}
								break;
							}
						}
					}
				}

				// Set default icons if above code fails to sets any
				if ( ! isset( $current_property_data['icon'] ) ) {
					$current_property_data['icon']       = INSPIRY_DIR_URI . '/images/map/single-family-home-map-icon.png';           // default icon
					$current_property_data['retinaIcon'] = INSPIRY_DIR_URI . '/images/map/single-family-home-map-icon@2x.png';  // default retina icon
				}

				$properties_map_data[] = $current_property_data;
			}

			wp_reset_postdata();
		}

		$map_stuff['modernHome']         = true;
		$map_stuff['closeIcon']          = INSPIRY_DIR_URI . '/images/map/close.png';
		$map_stuff['clusterIcon']        = INSPIRY_DIR_URI . '/images/map/cluster-icon.png';
		$map_stuff['infoBoxPlaceholder'] = get_inspiry_image_placeholder_url( 'property-thumb-image' );

		$properties_map_options['type'] = $settings['map_styles'];
		$styles_json                    = $settings['cta_map_style_json'];

		if ( ! empty( $styles_json ) ) {
			$properties_map_options['styles'] = wp_unslash( $styles_json );
		} else if ( $map_styles = get_option( 'inspiry_google_maps_styles', '' ) ) {
			$properties_map_options['styles'] = wp_unslash( $map_styles ); // Adding theme default if no style added in widget
		}

		$localize_map_data = array(
			'rheaPropertiesData' => json_encode( $properties_map_data ),
			'rheaMapStuff'       => json_encode( $map_stuff ),
			'rheaMapOptions'     => json_encode( $properties_map_options )
		);

		wp_localize_script( 'rhea-properties-google-map-js', 'googlePropertiesData', $localize_map_data );
		?>
        <div id="rhea-<?php echo $this->get_id(); ?>" class="rhea-listing-google-map"></div>
		<?php
	}

	public function render() {
		$this->rhea_google_properties_script();
		?>
        <div class="rhea_wrapper_properties_map">
            <div class="rhea-map-head">
				<?php
				$this->inject_content();
				?>
            </div>
        </div>
		<?php
	}
}