<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor Properties Filter.
 *
 * Elementor widget that creates a properties filter form for listing widgets
 *
 * @since 2.4.0
 */
class RHEA_Properties_Filter extends \Elementor\Widget_Base {

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );

		wp_register_script( 'rhea-properties-filter', RHEA_PLUGIN_URL . 'elementor/js/properties-filter.js', [ 'elementor-frontend' ], RHEA_VERSION, true );

		wp_localize_script( 'rhea-properties-filter', 'rheaPropertiesFilterStrings', array(
			'selected_text'  => esc_html__( 'Selected', RHEA_TEXT_DOMAIN ),
			'clear_all_text' => esc_html__( 'Clear All', RHEA_TEXT_DOMAIN )
		) );
	}

	/**
	 * Get widget name.
	 *
	 * Retrieve list widget name.
	 *
	 * @since  2.4.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name(): string {
		return 'rhea-properties-filter';
	}

	/**
	 * Get widget title.
	 *
	 * Retrieve list widget title.
	 *
	 * @since  2.4.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title(): string {
		return esc_html__( 'Properties Filter', RHEA_TEXT_DOMAIN );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve list widget icon.
	 *
	 * @since  2.4.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon(): string {
		return 'eicon-filter';
	}

	/**
	 * Get widget categories.
	 *
	 * Retrieve the list of categories the list widget belongs to.
	 *
	 * @since  2.4.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories(): array {
		return [ 'ultra-real-homes' ];
	}

	/**
	 * Enqueue required scripts
	 *
	 * Register and enqueue required scripts for this specific widget
	 *
	 * @since  2.4.0
	 * @access public
	 * @return array Widget script slugs.
	 */
	public function get_script_depends() {
		return [
			'rhea-properties-filter',
		];
	}

	/**
	 * Get widget keywords.
	 *
	 * Retrieve the list of keywords the list widget belongs to.
	 *
	 * @since  2.4.0
	 * @access public
	 * @return array Widget keywords.
	 */
	public function get_keywords(): array {
		return [ 'list', 'lists', 'listing', 'filter', 'filters', 'properties' ];
	}

	/**
	 * Get custom help URL.
	 *
	 * Retrieve a URL where the user can get more information about the widget.
	 *
	 * @since  2.4.0
	 * @access public
	 * @return string Widget help URL.
	 */
	public function get_custom_help_url(): string {
		return 'https://realhomes.io/documentation/properties-filter-widget/';
	}

	/**
	 * Whether the element returns dynamic content.
	 *
	 * Determine whether to cache the element output or not.
	 *
	 * @since  2.4.0
	 * @access protected
	 * @return bool Whether to cache the element output.
	 */
	protected function is_dynamic_content(): bool {
		return false;
	}

	/**
	 * Register filter widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since  2.4.0
	 * @access protected
	 */
	protected function register_controls(): void {

		$this->start_controls_section(
			'filter_settings',
			[
				'label' => esc_html__( 'Filter Settings', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'widget_title',
			[
				'label'       => esc_html__( 'Widget Title', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::TEXT,
				'placeholder' => esc_html__( 'Enter widget title', RHEA_TEXT_DOMAIN ),
				'default'     => esc_html__( 'Property Filters', RHEA_TEXT_DOMAIN )
			]
		);

		$this->add_control(
			'open_all_text',
			[
				'label'   => esc_html__( 'Open All Text', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Open All', RHEA_TEXT_DOMAIN )
			]
		);

		$this->add_control(
			'collapse_all_text',
			[
				'label'   => esc_html__( 'Collapse All Text', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Collapse All', RHEA_TEXT_DOMAIN )
			]
		);

		$this->add_control(
			'hide_empty_terms',
			[
				'label'        => esc_html__( 'Hide Empty Taxonomy Terms', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'hide',
			]
		);

		$this->add_control(
			'property_type_filter',
			[
				'label'        => esc_html__( 'Show Property Type Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'property_type_filter_view_state',
			[
				'label'        => esc_html__( 'Property Type Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'property_city_filter',
			[
				'label'        => esc_html__( 'Show Property City Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'property_city_filter_view_state',
			[
				'label'        => esc_html__( 'Property City Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'property_status_filter',
			[
				'label'        => esc_html__( 'Show Property Status Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'property_status_filter_view_state',
			[
				'label'        => esc_html__( 'Property Status Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'property_features_filter',
			[
				'label'        => esc_html__( 'Show Property Features Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'property_features_filter_view_state',
			[
				'label'        => esc_html__( 'Property Features Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'checkbox_view_limit',
			[
				'label'   => esc_html__( 'Checkboxes View Limit', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::NUMBER,
				'default' => 6
			]
		);

		$this->add_control(
			'price_filter',
			[
				'label'        => esc_html__( 'Show Price Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'price_filter_view_state',
			[
				'label'        => esc_html__( 'Price Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'area_filter',
			[
				'label'        => esc_html__( 'Show Area Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'area_unit',
			[
				'label'   => esc_html__( 'Area Unit', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => 'sqft'
			]
		);

		$this->add_control(
			'area_filter_view_state',
			[
				'label'        => esc_html__( 'Area Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'agents_filter',
			[
				'label'        => esc_html__( 'Show Agents Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'agents_filter_style',
			[
				'label'     => esc_html__( 'Agents Filter Style', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'thumbnail',
				'options'   => [
					'thumbnail' => esc_html__( 'Thumbnail', RHEA_TEXT_DOMAIN ),
					'checkbox'  => esc_html__( 'Checkbox', RHEA_TEXT_DOMAIN )
				],
				'condition' => [
					'agents_filter' => 'yes'
				]
			]
		);

		$this->add_control(
			'agent_thumbnail_size',
			[
				'label'     => esc_html__( 'Agent Thumbnail Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'thumbnail',
				'options'   => realhomes_get_available_image_sizes(),
				'condition' => [
					'agents_filter' => 'yes'
				]
			]
		);

		$this->add_control(
			'agent_filter_view_state',
			[
				'label'        => esc_html__( 'Agent Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'agency_filter',
			[
				'label'        => esc_html__( 'Show Agencies Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'agency_filter_style',
			[
				'label'     => esc_html__( 'Agency Filter Style', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'thumbnail',
				'options'   => [
					'thumbnail' => esc_html__( 'Thumbnail', RHEA_TEXT_DOMAIN ),
					'checkbox'  => esc_html__( 'Checkbox', RHEA_TEXT_DOMAIN )
				],
				'condition' => [
					'agency_filter' => 'yes'
				]
			]
		);

		$this->add_control(
			'agency_thumbnail_size',
			[
				'label'     => esc_html__( 'Agency Thumbnail Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'default'   => 'thumbnail',
				'options'   => realhomes_get_available_image_sizes(),
				'condition' => [
					'agency_filter' => 'yes'
				]
			]
		);

		$this->add_control(
			'agency_filter_view_state',
			[
				'label'        => esc_html__( 'Agency Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'bedroom_filter',
			[
				'label'        => esc_html__( 'Show Bedrooms Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'bedroom_filter_view_state',
			[
				'label'        => esc_html__( 'Bedroom Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'bathroom_filter',
			[
				'label'        => esc_html__( 'Show Bathrooms Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'bathroom_filter_view_state',
			[
				'label'        => esc_html__( 'Bathrooms Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'garage_filter',
			[
				'label'        => esc_html__( 'Show Garages Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'garages_filter_view_state',
			[
				'label'        => esc_html__( 'Garages Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'property_id_filter',
			[
				'label'        => esc_html__( 'Show Property ID Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'property_id_filter_view_state',
			[
				'label'        => esc_html__( 'Property ID Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
				'separator'    => 'after'
			]
		);

		$this->add_control(
			'additional_fields_filter',
			[
				'label'        => esc_html__( 'Show Additional Fields Filter', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'additional_fields_filter_view_state',
			[
				'label'        => esc_html__( 'Additional Section Default State', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Show', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'Hide', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes'
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'header_styles',
			[
				'label' => esc_html__( 'Widget Header Style', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => esc_html__( 'Title Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget h4.filters-heading'
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => esc_html__( 'Title Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget h4.filters-heading' => 'color: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'title_padding',
			[
				'label'      => esc_html__( 'Title Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget h4.filters-heading' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'collapse_text_typography',
				'label'    => esc_html__( 'Collapse All Text Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button > span .button-text'
			]
		);

		$this->add_control(
			'collapse_text_color',
			[
				'label'     => esc_html__( 'Collapse All Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button' => 'color: {{VALUE}};'
				],
			]
		);

		$this->add_control(
			'collapse_text_hover_color',
			[
				'label'     => esc_html__( 'Collapse All Text Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button:hover' => 'color: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'collapse_arrow_size',
			[
				'label'      => esc_html__( 'Arrow Size', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button > span i' => 'font-size: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'collapse_text_padding',
			[
				'label'      => esc_html__( 'Collapse Text Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'collapse_text_margin',
			[
				'label'      => esc_html__( 'Collapse Text Margin', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'collapse_arrow_color',
			[
				'label'     => esc_html__( 'Collapse All Arrow Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button > span i' => 'color: {{VALUE}};'
				],
			]
		);

		$this->add_control(
			'collapse_arrow_hover_color',
			[
				'label'     => esc_html__( 'Collapse All Text Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters .collapse-button:hover > span i' => 'color: {{VALUE}};'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'form_wrapper_styles',
			[
				'label' => esc_html__( 'Form Wrapper Style', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'filters_outer_wrapper_small_heading',
			[
				'label' => esc_html__( 'Outer Wrapper Section Styles', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::HEADING
			]
		);

		$this->add_control(
			'outer_wrapper_bg_color',
			[
				'label'     => esc_html__( 'Outer Wrapper BG Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filters-widget-wrap' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'outer_wrapper_box_shadow',
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filters-widget-wrap',
			]
		);

		$this->add_control(
			'outer_wrapper_padding',
			[
				'label'      => esc_html__( 'Outer Wrapper Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filters-widget-wrap' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'outer_wrapper_border_radius',
			[
				'label'      => esc_html__( 'Outer Wrapper Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filters-widget-wrap' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filters_inner_wrapper_small_heading',
			[
				'label' => esc_html__( 'Inner Wrapper Section Styles', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'inner_wrapper_bg_color',
			[
				'label'     => esc_html__( 'Inner Wrapper BG Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'inner_wrapper_box_shadow',
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .property-filters',
			]
		);

		$this->add_control(
			'inner_wrapper_padding',
			[
				'label'      => esc_html__( 'Inner Wrapper Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'separator'  => 'after'
			]
		);

		$this->add_control(
			'inner_wrapper_border_radius',
			[
				'label'      => esc_html__( 'Inner Wrapper Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .property-filters' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'form_sections_styles',
			[
				'label' => esc_html__( 'Form Sections Style', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'section_padding',
			[
				'label'      => esc_html__( 'Section Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sections_title_typography',
				'label'    => esc_html__( 'Title Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4'
			]
		);

		$this->add_control(
			'sections_title_color',
			[
				'label'     => esc_html__( 'Title Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'sections_title_hover_color',
			[
				'label'     => esc_html__( 'Title Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'sections_title_padding',
			[
				'label'      => esc_html__( 'Title Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'sections_title_selected_typography',
				'label'    => esc_html__( 'Selected Count Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 .counter'
			]
		);

		$this->add_control(
			'sections_title_selected_counter_color',
			[
				'label'     => esc_html__( 'Selected Count Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 .counter' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'widget_section_arrow_icon',
			[
				'label'       => esc_html__( 'Title Arrow Icon', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::ICONS,
				'recommended' => [
					'fa-solid' => [
						'caret-down',
						'angle-down',
						'angle-double-down',
						'arrow-down',
						'chevron-down',
						'hand-point-down',
						'long-arrow-alt-down'
					]
				],
			]
		);

		$this->add_control(
			'section_arrow_bg_position_horizontal',
			[
				'label'     => esc_html__( 'Arrow BG Position Horizontal', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min'  => -100,
						'max'  => 100
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon' => 'right: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'section_arrow_bg_position_vertical',
			[
				'label'     => esc_html__( 'Arrow BG Position Vertical', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => -100,
						'max' => 100
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon' => 'top: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'section_title_arrow_size',
			[
				'label'     => esc_html__( 'Title Arrow Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:after'                       => 'border-left-width: {{SIZE}}{{UNIT}}; border-right-width: {{SIZE}}{{UNIT}}; border-bottom-width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon i' => 'font-size: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'section_arrow_position_horizontal',
			[
				'label'     => esc_html__( 'Title Arrow Position Horizontal', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => -100,
						'max' => 100
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:after,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon i' => 'left: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'section_arrow_position_vertical',
			[
				'label'     => esc_html__( 'Title Arrow Position Vertical', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => -100,
						'max' => 100
					]
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:after,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon i' => 'top: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'section_arrow_border_radius',
			[
				'label'      => esc_html__( 'Arrow Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'section_arrow_bg_color',
			[
				'label'     => esc_html__( 'Title Arrow Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:before,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon' => 'background-color: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'section_arrow_hover_bg_color',
			[
				'label'     => esc_html__( 'Title Arrow Background Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover:before,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover span.section-control-icon' => 'background-color: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'section_title_arrow_color',
			[
				'label'     => esc_html__( 'Title Arrow Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:after'                                                 => 'border-bottom-color: {{VALUE}};',
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon i'                           => 'color: {{VALUE}};',
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon svg,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4 span.section-control-icon svg path'                     => 'fill: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'section_title_arrow_hover_color',
			[
				'label'     => esc_html__( 'Title Arrow Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover:after'                              => 'border-bottom-color: {{VALUE}};',
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover span.section-control-icon i'        => 'color: {{VALUE}};',
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover span.section-control-icon svg,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper h4:hover span.section-control-icon svg path'  => 'fill: {{VALUE}};'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'view_more_less_typography',
				'label'    => esc_html__( 'View More/Less Text Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .view-more, {{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .view-less'
			]
		);

		$this->add_control(
			'view_more_less_color',
			[
				'label'     => esc_html__( 'View More/Less Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .view-more,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .view-less' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'view_more_less_hover_color',
			[
				'label'     => esc_html__( 'View More/Less Text Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .view-more:hover,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .view-less:hover' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'section_bottom_separator_size',
			[
				'label'      => esc_html__( 'Bottom Separator Size', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper' => 'border-bottom-width: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'section_bottom_separator_color',
			[
				'label'     => esc_html__( 'Bottom Separator Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper' => 'border-bottom-color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'filter_checkbox_styles',
			[
				'label' => esc_html__( 'Checkbox Style', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'checkbox_section_padding',
			[
				'label'     => esc_html__( 'Checkboxes Section Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.additional-fields .additional-item:last-child > div' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_item_text_color',
			[
				'label'     => esc_html__( 'Checkbox Item Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'checkbox_items_typography',
				'label'    => esc_html__( 'Filter Items Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span, {{WRAPPER}} rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span'
			]
		);

		$this->add_control(
			'filter_checkbox_width',
			[
				'label'      => esc_html__( 'Checkbox Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span i' => 'width: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_height',
			[
				'label'      => esc_html__( 'Checkbox Height', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span i' => 'height: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_radius',
			[
				'label'     => esc_html__( 'Checkbox Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span i,
			{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span i' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_checkbox_position',
			[
				'label'      => esc_html__( 'Checkbox Position', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span i' => 'top: {{TOP}}{{UNIT}}; right: {{RIGHT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_checkbox_item_padding',
			[
				'label'      => esc_html__( 'Checkbox Item Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'default' => [
					'top' => 6,
					'right' => 10,
					'bottom' => 6,
					'left' => 42,
					'isLinked' => false
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .checkbox-wrap p' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				],
				'separator'  => 'after'
			]
		);

		$this->add_control(
			'filter_checkbox_border_color',
			[
				'label'     => esc_html__( 'Checkbox Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span i' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_bg_color',
			[
				'label'     => esc_html__( 'Checkbox Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span i' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_checked_border_color',
			[
				'label'     => esc_html__( 'Checkbox Checked Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span.active i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span.active i' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_checked_bg_color',
			[
				'label'     => esc_html__( 'Checkbox Checked Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span.active i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span.active i' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_checkbox_check_sign_size',
			[
				'label'      => esc_html__( 'Check Sign Size', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span.active i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span.active i' => 'font-size: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_checkbox_check_sign_position_horizontal',
			[
				'label'      => esc_html__( 'Check Position Horizontal', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => -100,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span.active i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span.active i' => 'text-indent: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_checkbox_check_sign_position_vertical',
			[
				'label'      => esc_html__( 'Check Position Vertical', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span.active i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span.active i' => 'line-height: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_checkbox_check_color',
			[
				'label'     => esc_html__( 'Checkbox Check Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-visible > span.active i,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .terms-list .items-view-more > span.active i' => 'color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();

		// Radio section styles
		$this->start_controls_section(
			'filter_radio_styles',
			[
				'label' => esc_html__( 'Radio Style', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'radio_filter_section_padding',
			[
				'label'     => esc_html__( 'Radio Section Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .filter-section.range-list,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.additional-fields .additional-item .radio-filter' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_radio_item_text_color',
			[
				'label'     => esc_html__( 'Radio Item Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap label' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'radio_items_typography',
				'label'    => esc_html__( 'Filter Items Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap label'
			]
		);

		$this->add_control(
			'radio_filter_item_padding',
			[
				'label'      => esc_html__( 'Filter Items Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap label' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_radio_button_width',
			[
				'label'      => esc_html__( 'Radio Button Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy' => 'width: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_radio_button_height',
			[
				'label'      => esc_html__( 'Radio Button Height', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy' => 'height: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_radio_position',
			[
				'label'      => esc_html__( 'Radio Button Position', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy' => 'top: {{TOP}}{{UNIT}}; right: {{RIGHT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_radio_border_color',
			[
				'label'     => esc_html__( 'Radio Button Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_radio_bg_color',
			[
				'label'     => esc_html__( 'Radio Button Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_radio_active_dot_width',
			[
				'label'      => esc_html__( 'Active Dot Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} rhea-properties-filter-widget .filter-wrapper .radio-wrap > input:checked + label .radio-fancy:after' => 'width: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_radio_active_dot_height',
			[
				'label'      => esc_html__( 'Active Dot Height', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} rhea-properties-filter-widget .filter-wrapper .radio-wrap > input:checked + label .radio-fancy:after' => 'height: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_radio_active_sign_position',
			[
				'label'      => esc_html__( 'Radio Active Dot Position', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy:after' => 'top: {{TOP}}{{UNIT}}; right: {{RIGHT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_radio_active_color',
			[
				'label'     => esc_html__( 'Radio Active Dot Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .radio-wrap .radio-fancy:after' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();

		// Agent Agency Section Styles
		$this->start_controls_section(
			'filter_agent_agency_styles',
			[
				'label' => esc_html__( 'Agent/Agency Section Styles', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'agent_agency_thumbnail_title_typography',
				'label'    => esc_html__( 'Title Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.thumbnail .item-content h5,
				.rhea-properties-filter-widget .filter-wrapper .posts-list .items-view-more .pt-item.thumbnail .item-content h5,
				{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span'
			]
		);

		$this->add_control(
			'agent_agency_thumbnail_title_color',
			[
				'label'     => esc_html__( 'Item Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.thumbnail .item-content h5,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_thumbnail_title_color_hover',
			[
				'label'     => esc_html__( 'Item Label Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.thumbnail .item-content h5:hover,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span:hover' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_thumbnail_item_padding',
			[
				'label'     => esc_html__( 'Items Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.thumbnail,
					{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_thumb_options_heading',
			[
				'label' => esc_html__( 'Thumbnail Options', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'agent_agency_thumbnail_width',
			[
				'label'      => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.thumbnail figure' => 'width: {{SIZE}}{{UNIT}}; height: auto;',
				]
			]
		);

		$this->add_control(
			'agent_agency_thumbnail_radius',
			[
				'label'      => esc_html__( 'Thumbnail Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.thumbnail figure' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_options_heading',
			[
				'label' => esc_html__( 'Checkbox Options', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_control(
			'agent_agency_checkbox_width',
			[
				'label'      => esc_html__( 'Checkbox Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'width: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_height',
			[
				'label'      => esc_html__( 'Checkbox Height', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'height: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_horizontal_position',
			[
				'label'      => esc_html__( 'Checkbox Horizontal Position', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'left: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_vertical_position',
			[
				'label'      => esc_html__( 'Checkbox Vertical Position', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'top: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_background',
			[
				'label'     => esc_html__( 'Checkbox Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_background_active',
			[
				'label'     => esc_html__( 'Checkbox Active Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox.active span i' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_border_size',
			[
				'label'      => esc_html__( 'Checkbox Border Size', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'border-width: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_radius',
			[
				'label'      => esc_html__( 'Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span:before' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_border_color',
			[
				'label'     => esc_html__( 'Checkbox Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_border_color_active',
			[
				'label'     => esc_html__( 'Checkbox Active Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox.active span i' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_check_size',
			[
				'label'      => esc_html__( 'Check Sign Size', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 100
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'font-size: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_check_line_height',
			[
				'label'      => esc_html__( 'Check Sign Line Height', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 10,
						'step' => 0.1
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'line-height: {{SIZE}};'
				]
			]
		);

		$this->add_control(
			'agent_agency_checkbox_check_color',
			[
				'label'     => esc_html__( 'Checkbox Check Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper .posts-list .items-visible .pt-item.checkbox span i' => 'color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();

		// Numbers Section Styles
		$this->start_controls_section(
			'filter_numbers_section_styles',
			[
				'label' => esc_html__( 'Numbers Sections Styles', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'filter_numbers_section_padding',
			[
				'label'      => esc_html__( 'Number Section Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .filter-section'  => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_width',
			[
				'label'      => esc_html__( 'Number Button Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'width: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_height',
			[
				'label'      => esc_html__( 'Number Button Height', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'height: {{SIZE}}{{UNIT}}; line-height: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_border_width',
			[
				'label'      => esc_html__( 'Number Button Border Width', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SLIDER,
				'range'      => [
					'px' => [
						'min'  => 0,
						'max'  => 200
					]
				],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'border-width: {{SIZE}}{{UNIT}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_radius',
			[
				'label'     => esc_html__( 'Number Button Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'filter_number_buttons_typography',
				'label'    => esc_html__( 'Number Button Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label'
			]
		);

		$this->add_control(
			'filter_number_buttons_border_color',
			[
				'label'     => esc_html__( 'Number Button Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_background_color',
			[
				'label'     => esc_html__( 'Number Button Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_text_color',
			[
				'label'     => esc_html__( 'Number Button Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span label' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_active_bordeer_color',
			[
				'label'     => esc_html__( 'Number Button Active Border Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span input:checked + label' => 'border-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_active_background_color',
			[
				'label'     => esc_html__( 'Number Button Active Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span input:checked + label' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_number_buttons_active_text_color',
			[
				'label'     => esc_html__( 'Number Button Active Text Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-filter-widget .filter-wrapper.radio-buttons .number-option-wrap > span input:checked + label' => 'color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();


		// Numbers Section Styles
		$this->start_controls_section(
			'filter_display_tags_section_styles',
			[
				'label' => esc_html__( 'Display Tags Section Styles', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE
			]
		);

		$this->add_control(
			'filters_display_tag_wrapper_margin',
			[
				'label'      => esc_html__( 'Tag Items Wrapper Margin', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'.elementor-widget .rhea-filters-display' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filters_display_tag_wrapper_padding',
			[
				'label'      => esc_html__( 'Tag Items Wrapper Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'.elementor-widget .rhea-filters-display' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filters_display_tag_wrapper_bg_color',
			[
				'label'     => esc_html__( 'Tag Items Wrapper Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'filters_display_tag_wrapper_border',
				'selector' => '.elementor-widget .rhea-filters-display',
			]
		);

		$this->add_control(
			'filters_display_tag_wrapper_shadow_heading',
			[
				'label' => esc_html__( 'Tag Items Wrapper Box Shadow', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'filters_display_tag_wrapper_shadow',
				'selector' => '.elementor-widget .rhea-filters-display'
			]
		);

		$this->add_control(
			'filters_display_tag_wrapper_border_radius',
			[
				'label'      => esc_html__( 'Tag Items Wrapper Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'.elementor-widget .rhea-filters-display' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filters_display_tag_item_margin',
			[
				'label'      => esc_html__( 'Tag Items Margin', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'.elementor-widget .rhea-filters-display > span' => 'Margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filters_display_tag_item_padding',
			[
				'label'      => esc_html__( 'Tag Items Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'.elementor-widget .rhea-filters-display > span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_tag_item_bg_color',
			[
				'label'     => esc_html__( 'Tag Item Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span' => 'background-color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Border::get_type(),
			[
				'name' => 'filter_tag_item_border',
				'selector' => '.elementor-widget .rhea-filters-display > span',
			]
		);

		$this->add_control(
			'filters_tag_item_wrapper_shadow_heading',
			[
				'label' => esc_html__( 'Tag Item Box Shadow', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'filters_tag_item_wrapper_shadow',
				'selector' => '.elementor-widget .rhea-filters-display > span'
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'filter_tag_item_value_typography',
				'label'    => esc_html__( 'Tag Value Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '.elementor-widget .rhea-filters-display > span'
			]
		);

		$this->add_control(
			'filter_tag_value_color',
			[
				'label'     => esc_html__( 'Tag Value Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'filter_tag_label_typography',
				'label'    => esc_html__( 'Tag Label Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '.elementor-widget .rhea-filters-display > span span'
			]
		);

		$this->add_control(
			'filter_tag_label_color',
			[
				'label'     => esc_html__( 'Tag Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span span' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_tag_item_close_x_size',
			[
				'label' => esc_html__( 'Tag Close X Size', RHEA_TEXT_DOMAIN ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100
					]
				],
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span i:before, .elementor-widget .rhea-filters-display > span i:after' => 'width: {{SIZE}}{{UNIT}};'
				]
			]
		);

		$this->add_control(
			'filter_tag_item_close_x_color',
			[
				'label'     => esc_html__( 'Tag Close X Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span i:before, .elementor-widget .rhea-filters-display > span i:after' => 'background-color: {{VALUE}};'
				]
			]
		);

		$this->add_control(
			'filter_tag_item_close_x_position',
			[
				'label'      => esc_html__( 'Tag Items Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors'  => [
					'.elementor-widget .rhea-filters-display > span i' => 'position: relative; top: {{TOP}}{{UNIT}}; right: {{RIGHT}}{{UNIT}}; bottom: {{BOTTOM}}{{UNIT}}; left: {{LEFT}}{{UNIT}};'
				]
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'filter_tag_clear_all_typography',
				'label'    => esc_html__( 'Filter Clear All Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY
				],
				'selector' => '.elementor-widget .rhea-filters-display>span.clear-all-filters.active'
			]
		);

		$this->add_control(
			'filter_tag_clear_all_color',
			[
				'label'     => esc_html__( 'Filter Clear All Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span.clear-all-filters.active' => 'color: {{VALUE}};',
				]
			]
		);

		$this->add_control(
			'filter_tag_clear_all_hover_color',
			[
				'label'     => esc_html__( 'Filter Clear All Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'.elementor-widget .rhea-filters-display > span.clear-all-filters.active:hover' => 'color: {{VALUE}};',
				]
			]
		);

		$this->end_controls_section();
	}

	/**
	 * Render list widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  2.4.0
	 * @access protected
	 */
	protected function render(): void {

		global $settings;
		$settings = $this->get_settings_for_display();

		// Get the post ID based on the context
		$post_id = null;

		if ( Elementor\Plugin::$instance->editor->is_edit_mode() ) {
			// In edit mode, get the current post ID
			$post_id = get_the_ID();
		} else {
			// Outside edit mode, ensure we have the correct post ID
			$post_id = get_the_ID();
		}

		// Ensure we have a valid post ID
		if ( ! $post_id ) {
			echo '<p class="alert alert-error"><strong>' . esc_html__( 'Error:', RHEA_TEXT_DOMAIN ) . '</strong> ' . esc_html__( 'Unable to retrieve content. Please ensure you are editing a valid page or post.', RHEA_TEXT_DOMAIN ) . '</p>';

			return;
		}

		// Get the page content using the post ID
		$content = Elementor\Plugin::$instance->frontend->get_builder_content_for_display( $post_id );

		// Define the property listing widget types
		$target_widgets = [
			'rhea-ultra-properties-widget-1',
			'elementor-property-listing-2'
		];

		// Define the search form widget types
		$target_form_widgets = [
			'rhea-search-form-1',
			'rhea-search-form-2'
		];

		// Check if any property listing widget is present
		$listing_widget_present = false;
		foreach ( $target_widgets as $widget ) {
			if ( strpos( $content, $widget ) !== false ) {
				$listing_widget_present = true;
				break;
			}
		}

		// Check if any search form widget is present
		$search_form_widget_present = false;
		foreach ( $target_form_widgets as $form_widget ) {
			if ( strpos( $content, $form_widget ) !== false ) {
				$search_form_widget_present = true;
				break;
			}
		}
		?>
        <div class="rhea-properties-filter-widget">
			<?php
			if ( ! empty( $settings['widget_title'] ) ) {
				echo '<h4 class="title filters-heading">' . esc_html( $settings['widget_title'] ) . ' <i class="fa-duotone fa-filters"></i></h4>';
			}

			rhea_get_template_part( 'elementor/widgets/properties-widget/filter-form' );
			?>
        </div>
		<?php
	}
}