<?php
/**
 * Elementor Properties V15 Widget
 *
 * @since 4.4.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Properties_Grid_V15 extends \Elementor\Widget_Base {
	private $is_rvr_enabled;

	public function __construct( array $data = [], ?array $args = null ) {
		parent::__construct( $data, $args );
		$this->is_rvr_enabled = rhea_is_rvr_enabled();
	}

	public function get_name() {
		return 'rhea-properties-grid-v15';
	}

	public function get_title() {
		return esc_html__( 'RH: Properties V15 (Grid)', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		// More classes for icons can be found at https://pojome.github.io/elementor-icons/
		return 'eicon-posts-grid rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-real-homes' ];
	}

	use RHEA_Properties_Partials;
	use RHEA_Properties_Action_buttons;

	protected function register_controls() {

		$grid_size_array = wp_get_additional_image_sizes();

		$prop_grid_size_array = array();
		foreach ( $grid_size_array as $key => $value ) {
			$str_rpl_key = ucwords( str_replace( "-", " ", $key ) );

			$prop_grid_size_array[ $key ] = $str_rpl_key . ' - ' . $value['width'] . 'x' . $value['height'];
		}

		unset( $prop_grid_size_array['partners-logo'] );
		unset( $prop_grid_size_array['property-detail-slider-thumb'] );
		unset( $prop_grid_size_array['post-thumbnail'] );
		unset( $prop_grid_size_array['agent-image'] );
		unset( $prop_grid_size_array['gallery-two-column-image'] );
		unset( $prop_grid_size_array['post-featured-image'] );

		$default_prop_grid_size = 'property-thumb-image';

		$allowed_html = array(
			'a'      => array(
				'href'  => array(),
				'title' => array()
			),
			'br'     => array(),
			'em'     => array(),
			'strong' => array(),
		);

		$this->start_controls_section(
			'ere_properties_section',
			[
				'label' => esc_html__( 'Properties', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'enable_search',
			[
				'label'        => esc_html__( 'Enable Property Search Filtering', RHEA_TEXT_DOMAIN ),
				'description'  => esc_html__( 'Enabling this will filter the property cards on search results page.', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'single_agent_agency_properties' => [ '', 'no' ],
				],
			]
		);

		$this->add_control(
			'single_agent_agency_properties',
			[
				'label'        => esc_html__( 'Display Properties for Agent/Agency Single Page', RHEA_TEXT_DOMAIN ),
				'description'  => esc_html__( 'Enabling this will display properties only on agent/agency single page based on agent/agency ID.', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
				'condition'    => [
					'enable_search' => [ '', 'no' ],
				],
			]
		);


		$this->add_control(
			'ere_property_grid_thumb_sizes',
			[
				'label'   => esc_html__( 'Grid Card Thumbnail Size', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'default' => $default_prop_grid_size,
				'options' => $prop_grid_size_array,
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'   => esc_html__( 'Number of Properties', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 60,
				'step'    => 1,
				'default' => 6,
			]
		);

		$get_agents = array();
		$agent_ids  = get_posts( array(
			'fields'         => 'ids',
			'posts_per_page' => -1,
			'post_type'      => 'agent'
		) );

		if ( ! empty( $agent_ids ) ) {
			foreach ( $agent_ids as $id ) {
				$get_agents["$id"] = get_the_title( $id );
			}
		}
		$this->add_control(
			'select_agent',
			[
				'label'       => esc_html__( 'Agent Based Properties', RHEA_TEXT_DOMAIN ),
				'type'        => \Elementor\Controls_Manager::SELECT2,
				'default'     => '',
				'options'     => $get_agents,
				'label_block' => true,
				'condition'   => [
					'single_agent_agency_properties' => [ '', 'no' ],
				],
			]
		);

		// Select controls for Custom Taxonomies related to Property
		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$options = [];
				$terms   = get_terms( $single_tax->name );

				if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
					foreach ( $terms as $term ) {
						$options[ $term->slug ] = $term->name;
					}
				}

				$this->add_control(
					$single_tax->name,
					[
						'label'       => $single_tax->label,
						'type'        => \Elementor\Controls_Manager::SELECT2,
						'multiple'    => true,
						'label_block' => true,
						'options'     => $options,
					]
				);
			}
		}

		// Sorting Controls
		$this->add_control(
			'orderby',
			[
				'label'   => esc_html__( 'Order By', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'date'       => esc_html__( 'Date', RHEA_TEXT_DOMAIN ),
					'price'      => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
					'title'      => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
					'menu_order' => esc_html__( 'Menu Order', RHEA_TEXT_DOMAIN ),
					'rand'       => esc_html__( 'Random', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'date',
			]
		);

		$this->add_control(
			'order',
			[
				'label'   => esc_html__( 'Order', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'asc'  => esc_html__( 'Ascending', RHEA_TEXT_DOMAIN ),
					'desc' => esc_html__( 'Descending', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'desc',
			]
		);

		$this->add_control(
			'featured_properties_condition',
			[
				'label'   => esc_html__( 'Featured Properties', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'normal'             => esc_html__( 'Normal', RHEA_TEXT_DOMAIN ),
					'show_featured_only' => esc_html__( 'Show Featured Only', RHEA_TEXT_DOMAIN ),
					'hide_featured'      => esc_html__( 'Hide Featured', RHEA_TEXT_DOMAIN ),
				],
				'default' => 'normal',
			]
		);

		$this->add_control(
			'skip_sticky_properties',
			[
				'label'        => esc_html__( 'Skip Sticky Properties', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'offset',
			[
				'label'   => esc_html__( 'Offset or Skip From Start', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::NUMBER,
				'default' => '0',
			]
		);


		$this->add_control(
			'ere_show_property_status',
			[
				'label'        => esc_html__( 'Show Property Status', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'ere_show_property_media_count',
			[
				'label'        => esc_html__( 'Show Property Media Count', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);


		$this->add_control(
			'ere_enable_fav_properties',
			[
				'label'        => esc_html__( 'Show Add To Favourite Button', RHEA_TEXT_DOMAIN ),
				'description'  => wp_kses( __( '<strong>Important:</strong> Make sure to select <strong>Show</strong> in Customizer <strong>Dashboard > My Favorites (Add to Favorites Button on Property Detail Page)</strong> settings. ', RHEA_TEXT_DOMAIN ), $allowed_html ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'ere_enable_compare_properties',
			[
				'label'        => esc_html__( 'Show Add To Compare Button  ', RHEA_TEXT_DOMAIN ),
				'description'  => wp_kses( __( '<strong>Important:</strong> Make sure to select <strong>Enable</strong> in Customizer <strong>Floating Features (Compare Properties)</strong> settings. ', RHEA_TEXT_DOMAIN ), $allowed_html ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'show_date',
			[
				'label'        => esc_html__( 'Show Date', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'ere_choose_post_time_formats',
			[
				'label'     => esc_html__( 'Post Date Format', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'dashboard' => esc_html__( 'Default( Dashboard > Settings > General )', RHEA_TEXT_DOMAIN ),
					'default'   => esc_html__( 'Custom', RHEA_TEXT_DOMAIN ),
					'human'     => esc_html__( 'Human Readable', RHEA_TEXT_DOMAIN ),
				],
				'default'   => 'default',
				'condition' => [
					'show_date' => 'yes',
				],

			]
		);

		$this->add_control(
			'ere_choose_post_time_format_option',
			[
				'label'      => esc_html__( 'Post Date Format Options', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::SELECT,
				'options'    => [
					'F j, Y'           => esc_html__( 'November 6, 2010', RHEA_TEXT_DOMAIN ),
					'F j, Y g:i a'     => esc_html__( 'November 6, 2010 12:50 am', RHEA_TEXT_DOMAIN ),
					'F, Y'             => esc_html__( 'November, 2010', RHEA_TEXT_DOMAIN ),
					'g:i a'            => esc_html__( '12:50 am', RHEA_TEXT_DOMAIN ),
					'l, F jS, Y'       => esc_html__( 'Saturday, November 6th, 20', RHEA_TEXT_DOMAIN ),
					'M j, Y @ G:i'     => esc_html__( 'Nov 6, 2010 @ 0:50', RHEA_TEXT_DOMAIN ),
					'Y/m/d \a\t g:i A' => esc_html__( '2010/11/06 at 12:50 AM', RHEA_TEXT_DOMAIN ),
					'Y/m/d \a\t g:ia'  => esc_html__( '2010/11/06 at 12:50am', RHEA_TEXT_DOMAIN ),
					'Y/m/d g:i:s A'    => esc_html__( '2010/11/06 12:50:48 AM', RHEA_TEXT_DOMAIN ),
					'Y/m/d'            => esc_html__( ' 2010/11/06', RHEA_TEXT_DOMAIN ),
				],
				'default'    => 'F j, Y',
				'conditions' => [
					'relation' => 'and',
					'terms'    => [
						[
							'name'     => 'show_date',
							'operator' => '==',
							'value'    => 'yes'
						],
						[
							'name'     => 'ere_choose_post_time_formats',
							'operator' => '==',
							'value'    => 'default'
						],
					]
				],
			]
		);

		$this->add_control(
			'show_address',
			[
				'label'        => esc_html__( 'Show Property Address', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'ere_show_property_type',
			[
				'label'        => esc_html__( 'Show Property Type', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'yes',

			]
		);

		$this->add_control(
			'show_animate_contents',
			[
				'label'        => esc_html__( 'Show Animate Contents (Editor Mode Only)', RHEA_TEXT_DOMAIN ),
				'description'  => esc_html__( 'Only for Elementor Editor to stylize the contents', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'ere_properties_labels',
			[
				'label' => esc_html__( 'Property Labels', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);

		$this->action_buttons_labels();
		$this->inline_action_buttons_labels();

		$this->add_control(
			'ere_property_rvt_date_added_label',
			[
				'label'   => esc_html__( 'Date Added Label', RHEA_TEXT_DOMAIN ),
				'type'    => \Elementor\Controls_Manager::TEXT,
				'default' => esc_html__( 'Added:', RHEA_TEXT_DOMAIN ),
			]
		);

		$this->end_controls_section();

		$this->meta_icons_controls( $this->is_rvr_enabled );

		$this->start_controls_section(
			'grid_settings_section',
			[
				'label' => esc_html__( 'Grid Settings', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'grid_details_padding',
			[
				'label'      => esc_html__( 'Grid Details Padding', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rhea-property-detail-box' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'grid_border_radius',
			[
				'label'      => esc_html__( 'Grid Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-grid-v15-wrapper .rhea-grid-item'     => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					'{{WRAPPER}} .rhea-properties-grid-v15-wrapper .rhea-property-link' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'grid_details_border_radius',
			[
				'label'      => esc_html__( 'Grid Details Border Radius', RHEA_TEXT_DOMAIN ),
				'type'       => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%', 'em' ],
				'selectors'  => [
					'{{WRAPPER}} .rhea-properties-grid-v15-wrapper .rhea-property-detail' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'grid_detail_background',
			[
				'label'     => esc_html__( 'Grid Details Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-properties-grid-v15-wrapper .rhea-property-detail' => 'background: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->properties_tags_controls();

		$this->start_controls_section(
			'property_title_section',
			[
				'label' => esc_html__( 'Property Title', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_title_typo',
				'label'    => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea-property-detail h3 a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_wide_title_typo',
				'label'    => esc_html__( 'Wide Column Title', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea-property-detail.span-wide h3 a',
			]
		);

		$this->add_control(
			'property_title_color',
			[
				'label'     => esc_html__( 'Title Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-property-detail h3 a' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'property_title_color_hover',
			[
				'label'     => esc_html__( 'Title Hover Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea-property-detail h3 a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'title_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-property-detail h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'title_wide_margin_bottom',
			[
				'label'     => esc_html__( 'Wide Column Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-property-detail.span-wide h3' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
		$this->start_controls_section(
			'property_address_section',
			[
				'label' => esc_html__( 'Property Address', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_address_typography',
				'label'    => esc_html__( 'Address', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea_address_ultra a',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_wide_address_typography',
				'label'    => esc_html__( 'Wide Column Address', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .span-wide .rhea_address_ultra a',
			]
		);

		$this->add_control(
			'property_address_color',
			[
				'label'     => esc_html__( 'Address Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea_address_ultra a' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_control(
			'property_address_color_hover',
			[
				'label'     => esc_html__( 'Address Hover color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea_address_ultra a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_address_icon_color',
			[
				'label'     => esc_html__( 'Address Icon Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea_address_ultra .rhea_ultra_address_pin svg' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_address_icon_inner_color',
			[
				'label'     => esc_html__( 'Address Icon Inner Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea_address_ultra .rhea_ultra_address_pin svg .line-st1' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'address_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rhea_address_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'address_wide_margin_bottom',
			[
				'label'     => esc_html__( 'Wide Column Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .span-wide .rhea_address_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
		$this->start_controls_section(
			'property_date_section',
			[
				'label' => esc_html__( 'Property Date', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_date_typography',
				'label'    => esc_html__( 'Date', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea-date-wrapper',
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_wide_date_typography',
				'label'    => esc_html__( 'Wide Column Date', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .span-wide .rhea-date-wrapper',
			]
		);
		$this->add_control(
			'rhea_added_date_label_color',
			[
				'label'     => esc_html__( 'Added Date Label Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rvr_card_info_wrap .added-date .added-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_added_date_color',
			[
				'label'     => esc_html__( 'Added Date Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rvr_card_info_wrap .added-date' => 'color: {{VALUE}}',
				],
			]
		);
		$this->add_responsive_control(
			'date_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rhea-date-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'date_wide_margin_bottom',
			[
				'label'     => esc_html__( 'Wide Column Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .span-wide .rhea-date-wrapper' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
		$this->start_controls_section(
			'property_price_section',
			[
				'label' => esc_html__( 'Property Price', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_prefix_typography',
				'label'    => esc_html__( 'Price Prefix (i.e From)', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .ere-price-prefix',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_typography',
				'label'    => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .ere-price-display,{{WRAPPER}} .property-current-price',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_wide_price_typography',
				'label'    => esc_html__( 'Wide Column Price', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .span-wide .ere-price-display,{{WRAPPER}} .span-wide .property-current-price',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_old_price_typography',
				'label'    => esc_html__( 'Old Price', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .property-old-price',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_price_postfix_typography',
				'label'    => esc_html__( 'Price Postfix (i.e Monthly)', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .ere-price-postfix',
			]
		);

		$this->add_control(
			'price-meta-wrapper-flex',
			[
				'label'     => esc_html__( 'Price Meta Wrapper Flex', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'initial' => esc_html__( 'Initial', RHEA_TEXT_DOMAIN ),
					'wrap'    => esc_html__( 'Wrap', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} .price-meta-wrapper' => 'flex-wrap: {{VALUE}};',
				],
				'default'   => 'initial',
			]
		);

		$this->add_control(
			'show_price_prefix',
			[
				'label'     => esc_html__( 'Show Price Prefix', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'none'        => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
					'inline-flex' => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} p.rh_prop_card__price_ultra .ere-price-prefix' => 'display: {{VALUE}};',
				],
				'default'   => 'none',
			]
		);

		$this->add_control(
			'show_price_slash',
			[
				'label'        => esc_html__( 'Show Price Postfix Separator (i.e "/")', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'none'        => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
					'inline-flex' => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} p.rh_prop_card__price_ultra .ere-price-slash' => 'display: {{VALUE}};',
				],
				'default'   => 'none',
			]
		);

		$this->add_control(
			'show_old_price',
			[
				'label'     => esc_html__( 'Show Old Price', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'none'        => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
					'inline-flex' => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} p.rh_prop_card__price_ultra .property-old-price' => 'display: {{VALUE}};',
				],
				'default'   => 'none',
			]
		);

		$this->add_control(
			'show_price_postfix',
			[
				'label'     => esc_html__( 'Show Price Postfix', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SELECT,
				'options'   => [
					'none'        => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
					'inline-flex' => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				],
				'selectors' => [
					'{{WRAPPER}} p.rh_prop_card__price_ultra .ere-price-postfix' => 'display: {{VALUE}};',
				],
				'default'   => 'none',
			]
		);

		$this->add_control(
			'rhea_price_prefix_color',
			[
				'label'     => esc_html__( 'Price Prefix Color (i.e From)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .ere-price-prefix' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_price_text_color',
			[
				'label'     => esc_html__( 'Price Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .ere-price-display' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_old_price_color',
			[
				'label'     => esc_html__( 'Old Price Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .property-old-price' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_price_postfix_color',
			[
				'label'     => esc_html__( 'Price Postfix Color (i.e Monthly)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .ere-price-postfix' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_price_separator_color',
			[
				'label'     => esc_html__( 'Price Separator Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .ere-price-slash' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'price_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh_prop_card__priceLabel_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'price_wide_margin_bottom',
			[
				'label'     => esc_html__( 'Wide Column Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .span-wide .rh_prop_card__priceLabel_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
		$this->start_controls_section(
			'property_meta_section',
			[
				'label' => esc_html__( 'Property Meta Icons', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_labels_typography',
				'label'    => esc_html__( 'Meta Labels', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea_ultra_meta_box .figure',
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'property_meta_figures_typography',
				'label'    => esc_html__( 'Figures postfix (i.e sqft)', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rhea_ultra_meta_box .label',
			]
		);

		$this->add_control(
			'rhea_meta_icon_fa_color',
			[
				'label'     => esc_html__( 'Meta Icons FontAwesome', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} i' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_meta_icon_fill_dark_color',
			[
				'label'     => esc_html__( 'Meta Icons Fill Dark', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-ultra-dark' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_meta_icon_fill_light_color',
			[
				'label'     => esc_html__( 'Meta Icons Fill Light', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-ultra-light' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_meta_icon_stroke_dark_color',
			[
				'label'     => esc_html__( 'Meta Icons Stroke Dark', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-ultra-stroke-dark' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_meta_icon_stroke_light_color',
			[
				'label'     => esc_html__( 'Meta Icons Stroke Light', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-ultra-stroke-light' => 'stroke: {{VALUE}}',
				],
			]
		);


		$this->add_control(
			'rhea_meta_figure_color',
			[
				'label'     => esc_html__( 'Meta Figures', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea_ultra_meta_box .figure' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'rhea_meta_figure_postfix_color',
			[
				'label'     => esc_html__( 'Meta Figures postfix (i.e sqft)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rhea_ultra_meta_box .label' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'meta_margin_bottom',
			[
				'label'     => esc_html__( 'Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh_prop_card_meta_wrap_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'meta_wide_margin_bottom',
			[
				'label'     => esc_html__( 'Wide Column Margin Bottom', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .span-wide .rh_prop_card_meta_wrap_ultra' => 'margin-bottom: {{SIZE}}{{UNIT}};',
				],
			]
		);


		$this->end_controls_section();
		$this->action_button_controls_v2();

	}

	protected function render() {
		global $settings, $widget_id, $properties_query, $paged;

		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();
		$meta_query = array();

		$paged = 1;
		if ( get_query_var( 'paged' ) ) {
			$paged = get_query_var( 'paged' );
		} else if ( get_query_var( 'page' ) ) { // if is static front page
			$paged = get_query_var( 'page' );
		}

		if ( $settings['offset'] ) {
			$offset = $settings['offset'] + ( $paged - 1 ) * $settings['posts_per_page'];
		} else {
			$offset = '';
		}

		$properties_args = array(
			'post_type'      => 'property',
			'posts_per_page' => $settings['posts_per_page'],
			'order'          => $settings['order'],
			'offset'         => $offset,
			'post_status'    => 'publish',
			'paged'          => $paged,
		);

		// Sticky property filtering
		if ( $settings['skip_sticky_properties'] !== 'yes' ) {
			$meta_query[] = [
				'key'     => 'REAL_HOMES_sticky',
				'compare' => 'EXISTS',
			];
		}

		// Featured properties condition
		if ( ! empty( $settings['featured_properties_condition'] ) ) {
			$condition = $settings['featured_properties_condition'];

			if ( 'show_featured_only' === $condition ) {
				$meta_query[] = [
					'key'     => 'REAL_HOMES_featured',
					'value'   => 1,
					'compare' => '=',
					'type'    => 'NUMERIC',
				];
			} else if ( 'hide_featured' === $condition ) {
				$meta_query[] = [
					'key'     => 'REAL_HOMES_featured',
					'value'   => 1,
					'compare' => '!=',
					'type'    => 'NUMERIC',
				];
			}
		}

		$select_agent = $settings['select_agent'];
		if ( 'yes' === $settings['single_agent_agency_properties'] || ! empty( $select_agent ) ) {
			$agent_id = get_the_ID();
			if ( ! empty( $select_agent ) ) {
				$agent_id = $select_agent;
			}
			$meta_query[] = array(
				'key'     => 'REAL_HOMES_agents',
				'value'   => $agent_id,
				'compare' => '=',
			);
		}

		// Add meta_query if not empty
		if ( ! empty( $meta_query ) ) {
			$properties_args['meta_query'] = $meta_query;
		}

		// Sorting
		if ( 'price' === $settings['orderby'] ) {
			$properties_args['orderby']  = 'meta_value_num';
			$properties_args['meta_key'] = 'REAL_HOMES_property_price';
		} else {
			// for date, title, menu_order and rand
			$properties_args['orderby'] = $settings['orderby'];
		}

		// Filter based on custom taxonomies
		$property_taxonomies = get_object_taxonomies( 'property', 'objects' );
		if ( ! empty( $property_taxonomies ) && ! is_wp_error( $property_taxonomies ) ) {
			foreach ( $property_taxonomies as $single_tax ) {
				$setting_key = $single_tax->name;
				if ( ! empty( $settings[ $setting_key ] ) ) {
					$properties_args['tax_query'][] = [
						'taxonomy' => $setting_key,
						'field'    => 'slug',
						'terms'    => $settings[ $setting_key ],
					];
				}
			}

			if ( isset( $properties_args['tax_query'] ) && count( $properties_args['tax_query'] ) > 1 ) {
				$properties_args['tax_query']['relation'] = 'AND';
			}
		}

		if ( 'yes' === $settings['enable_search'] ) {
			$properties_args                   = apply_filters( 'real_homes_search_parameters', $properties_args );
			$properties_args['posts_per_page'] = $settings['posts_per_page'];
		}

		$properties_args = apply_filters( 'rhea_modern_properties_widget', $properties_args );

		if ( is_tax() ) {
			global $wp_query;
			$properties_args = array_merge( $wp_query->query_vars, $properties_args );
		}

		$properties_query = new WP_Query( $properties_args );

		if ( \Elementor\Plugin::$instance->editor->is_edit_mode() && 'yes' === $settings['show_animate_contents'] ) {
			$show_contents = ' animate-in ';
		} else {
			$show_contents = '';
		}

		if ( $properties_query->have_posts() ) {
			?>
            <div class="rhea-properties-grid-v15-wrapper style-1">
				<?php
				$post_count  = 0; // Counter for the posts
				$total_posts = $properties_query->post_count; // Total number of posts in the loop

				while ( $properties_query->have_posts() ) {
					$properties_query->the_post();
					$post_count++;

					$property_id = get_the_ID();

					if ( $post_count == 1 || 0 == $post_count % 6 || 0 == $post_count % 7 ) {
						$classes = 'span-wide bottom-on-devices left';
					} else {
						$classes = 'bottom';
					}
					?>
                    <div class="rhea-grid-item rhea-trigger-animate animate-item-<?php echo esc_attr( $widget_id ) ?> ">
                        <a href="<?php the_permalink() ?>" class="rhea-property-link">
							<?php echo get_the_post_thumbnail( $property_id, 'full' ); ?>
                        </a>
						<?php rhea_render_properties_tags( $property_id, $settings, 'rhea-animate-item top deanimate-top-sm-devices' . $show_contents ); ?>
                        <div class="rhea-property-detail rhea-animate-item deanimate-bottom-sm-devices <?php echo esc_attr( $classes . $show_contents ) ?>">
                            <div class="rhea-property-detail-box">
                                <h3><a href="<?php the_permalink( $property_id ); ?>"><?php the_title() ?></a></h3>
								<?php
								rhea_get_template_part( 'assets/partials/ultra/address' );
								if ( 'yes' === $settings['show_date'] ) {
									?>
                                    <div class="rhea-date-wrapper">
                                        <span class="added-title"><?php echo esc_html( $settings['ere_property_rvt_date_added_label'] ); ?></span>
										<?php echo get_the_date(); ?>
                                    </div>
									<?php
								}
								?>
                                <div class="price-meta-wrapper">
									<?php
									rhea_get_template_part( 'assets/partials/ultra/price' );

									$this->render_meta_icons( $property_id, $settings );
									?>
                                </div>
                            </div>
                            <div class="rhea-action-button-box">
                                <div class="rhea-ultra-action-buttons rh-ultra-action-light hover-light">
									<?php
									rhea_render_favourite_button( $property_id, $settings, '/common/images/icons/ultra-favourite.svg', 'ultra' );
									rhea_render_compare_button( $property_id, $settings );
									?>
                                </div>
								<?php rhea_render_media_count_button( $property_id, $widget_id, $settings, true ); ?>
                            </div>
                        </div>
                    </div>
					<?php
				}
				?>
            </div>
			<?php
		}
	}
}