<?php
/**
 * Single Property Features Widget V2.
 *
 * @since 2.4.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Features_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	// Static properties for reusable data.
	private static $features_display;
	private static $default_icon;
	private static $property_id;
	private static $features;

	public function __construct( $data = [], $args = null ) {
		parent::__construct( $data, $args );

		// Initialize static properties if not already set.
		if ( is_null( self::$features_display ) ) {
			self::$features_display = get_option( 'inspiry_property_features_display', 'link' );
		}

		if ( is_null( self::$default_icon ) ) {
			self::$default_icon = $this->generate_default_icon();
		}

		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}

		if ( is_null( self::$features ) ) {
			self::$features = get_the_terms( self::$property_id, 'property-feature' );
		}
	}

	public function get_name() {
		return 'rh-single-property-features-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Features V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-bullet-list rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	protected function register_controls() {

		$this->title_section_content_tab_controls( esc_html__( 'Features', RHEA_TEXT_DOMAIN ) );

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'features_colors_section',
			[
				'label' => esc_html__( 'Features', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'features_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-features-v2 .feature-item-label',
			]
		);

		$this->add_control(
			'feature_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item-label, {{WRAPPER}} .rh-single-property-features-v2 .feature-item-label a' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'feature_link_hover_color',
			[
				'label'     => esc_html__( 'Color Hover', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item-label a:hover' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'features_column_gap',
			[
				'label'     => esc_html__( 'Items Column Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2-list' => '--feature-item-column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'features_row_gap',
			[
				'label'     => esc_html__( 'Items Row Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2-list' => '--feature-item-row-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'features_width',
			[
				'label'     => esc_html__( 'Items Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 500,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item' => 'width: calc({{SIZE}}{{UNIT}} - var(--feature-item-column-gap));',
				],
			]
		);

		$this->add_control(
			'icon_section_heading',
			[
				'label' => esc_html__( 'SVG Icons', RHEA_TEXT_DOMAIN ),
				'type'  => \Elementor\Controls_Manager::HEADING,
			]
		);

		$this->add_control(
			'svg_color',
			[
				'label'     => esc_html__( 'Fill', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item svg,
					 {{WRAPPER}} .rh-single-property-features-v2 .feature-item svg path,
					 {{WRAPPER}} .rh-single-property-features-v2 .feature-item svg rect' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'svg_stroke',
			[
				'label'     => esc_html__( 'Stroke', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item svg path,
					{{WRAPPER}} .rh-single-property-features-v2 .feature-item svg rect' => 'stroke: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label'     => esc_html__( 'Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-features-v2 .feature-item img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {

		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() ) {
			return;
		}

		$settings  = $this->get_settings_for_display();
		$widget_id = $this->get_id();
		?>
        <div id="rh-single-property-features-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-features-v2 <?php realhomes_printable_section( 'features' ); ?>">
			<?php
			if ( ! empty( self::$features ) && ! is_wp_error( self::$features ) ) {
				if ( ! empty( $settings['section_title'] ) ) {
					?>
                    <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
					<?php
				}
				?>
                <div class="rh-single-property-features-v2-inner">
					<?php


					echo '<ul class="rh-single-property-features-v2-list">';

					foreach ( self::$features as $feature ) {
						$feature_id = $feature->term_id;
						$icon_url   = $this->get_feature_icon_url( $feature_id );

						echo '<li class="feature-item" id="feature-id-' . esc_attr( $feature_id ) . '">';
						echo $this->get_feature_icon_markup( $icon_url, $feature->name );
						echo $this->get_feature_label_markup( $feature );
						echo '</li>';
					}
					echo '</ul>';
					?>
                </div>
				<?php
			} else {
				rhea_print_no_result_for_editor();
			}
			?>
        </div>
		<?php
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}

	private function generate_default_icon() {
		ob_start();
		?>
        <svg xmlns="http://www.w3.org/2000/svg" width="26px" height="26px" viewBox="0 0 22 22" class="responsive-icon">
            <path d="M11 .8C16.7.8 21.3 5.4 21.3 11S16.7 21.2 11 21.2C5.3 21.2.8 16.7.8 11S5.3.8 11 .8z" opacity=".3"></path>
            <path class="rh-ultra-light rh-ultra-stroke-dark" d="M11 .8C16.7.8 21.3 5.4 21.3 11S16.7 21.2 11 21.2C5.3 21.2.8 16.7.8 11S5.3.8 11 .8z" fill="none" stroke="#000" stroke-width="1.5"></path>
            <path class="rh-ultra-stroke-dark" d="m5.7 12 3 3 7-8" fill="none" stroke="#000" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"></path>
        </svg>
		<?php
		return ob_get_clean();
	}

	public function get_feature_icon_url( $feature_id ) {
		$icon_id = get_term_meta( $feature_id, 'inspiry_property_feature_icon', true );

		return $icon_id ? wp_get_attachment_url( $icon_id ) : false;
	}

	public function get_feature_icon_markup( $icon_url, $feature ) {
		if ( $icon_url ) {
			if ( preg_match( '/\.svg$/', $icon_url ) ) {
				$svg_file = wp_remote_get( $icon_url );
				if ( is_array( $svg_file ) && ! is_wp_error( $svg_file ) ) {
					$svg_content   = wp_remote_retrieve_body( $svg_file );
					$sanitized_svg = ( new RealHomes_Sanitize_Svg() )->sanitize( $svg_content );

					if ( false !== $sanitized_svg ) {
						return $sanitized_svg;
					}
				}
			} else {
				return '<img class="property-feature-icon property-feature-icon-image" src="' . esc_url( $icon_url ) . '" alt="' . esc_attr( $feature ) . '">';
			}
		}

		return self::$default_icon;
	}

	public function get_feature_label_markup( $feature ) {
		$label = esc_html( $feature->name );

		if ( 'link' === self::$features_display ) {
			$label = sprintf( '<a href="%s">%s</a>', esc_url( get_term_link( $feature->slug, 'property-feature' ) ), $label );
		}

		return sprintf( '<span class="feature-item-label">%s</span>', $label );
	}
}