<?php
/**
 * Single Property Floor Plans Widget V2.
 *
 * @since 2.4.0
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class RHEA_Single_Property_Floor_Plans_V2 extends \Elementor\Widget_Base {

	use RHEA_Single_Property_V2_Title_Section_Trait;

	// Static properties for reusable data.
	private static $property_id;

	public function __construct( $data = [], $args = null ) {
		parent::__construct( $data, $args );

		if ( is_null( self::$property_id ) ) {
			self::$property_id = $this->initialize_property_id();
		}

		wp_register_script( 'rh-single-property-floor-plans-v2', RHEA_PLUGIN_URL . 'elementor/js/rh-single-property-floor-plans-v2.js', [ 'elementor-frontend' ], RHEA_VERSION, true );
	}

	public function get_name() {
		return 'rh-single-property-floor-plans-v2';
	}

	public function get_title() {
		return esc_html__( 'RH: Single Property Floor Plans V2', RHEA_TEXT_DOMAIN );
	}

	public function get_icon() {
		return 'eicon-tabs rh-ultra-widget';
	}

	public function get_categories() {
		return [ 'ultra-realhomes-single-property' ];
	}

	public function get_script_depends() {
		return [ 'rh-single-property-floor-plans-v2' ];
	}

	protected function register_controls() {

		$this->title_section_content_tab_controls( esc_html__( 'Property Floor Plans', RHEA_TEXT_DOMAIN ) );

		$this->title_section_style_tab_controls();

		$this->start_controls_section(
			'container_section',
			[
				'label' => esc_html__( 'Container', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'container_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'container_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'container_content_gap',
			[
				'label'     => esc_html__( 'Content Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'container_bg_color',
			[
				'label'     => esc_html__( 'Background', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs' => 'background: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'container_box_shadow',
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'tabs_section',
			[
				'label' => esc_html__( 'Tabs', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tabs_width',
			[
				'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list' => 'width: {{SIZE}}%;',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_height',
			[
				'label'     => esc_html__( 'Height', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 300,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li,
				     {{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li img' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_gap',
			[
				'label'     => esc_html__( 'Gap', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 150,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'disable_tabs_image',
			[
				'label'        => esc_html__( 'Disable Image', RHEA_TEXT_DOMAIN ),
				'type'         => \Elementor\Controls_Manager::SWITCHER,
				'label_on'     => esc_html__( 'Yes', RHEA_TEXT_DOMAIN ),
				'label_off'    => esc_html__( 'No', RHEA_TEXT_DOMAIN ),
				'return_value' => 'yes',
				'default'      => 'no',
			]
		);

		$this->add_control(
			'tabs_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'disable_tabs_image' => 'yes',
				],
			]
		);

		$this->add_control(
			'active_tabs_bg_color',
			[
				'label'     => esc_html__( 'Background Color (Active)', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li.active' => 'background-color: {{VALUE}}',
				],
				'condition' => [
					'disable_tabs_image' => 'yes',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'tabs_box_shadow',
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li',
			]
		);

		$this->add_control(
			'tabs_label_heading',
			[
				'label'     => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'tabs_title_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li span' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_title_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li span' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'tabs_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li span',
			]
		);

		$this->add_control(
			'tabs_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li span' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'tabs_title_bg_color',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li span' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'name'     => 'tabs_title_box_shadow',
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li span',
			]
		);

		$this->add_control(
			'active_tab_arrow_heading',
			[
				'label'     => esc_html__( 'Active Tab Arrow', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'active_tab_arrow_size',
			[
				'label'     => esc_html__( 'Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 30,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li.active:after' => 'border-top-width: {{SIZE}}{{UNIT}}; border-bottom-width: {{SIZE}}{{UNIT}}; border-left-width: calc( {{SIZE}}{{UNIT}} + 3px);' . ( is_rtl() ? 'left: -{{SIZE}}{{UNIT}};' : 'right: -{{SIZE}}{{UNIT}};' ),
				],
			]
		);

		$this->add_control(
			'active_tab_arrow_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-list li.active:after' => 'border-left-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'tabs_separator_section',
			[
				'label' => esc_html__( 'Separator', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tabs_separator_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);


		$this->add_responsive_control(
			'tabs_separator_width',
			[
				'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator'                                                                             => 'width: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator:before, {{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator:after' => 'width: calc({{SIZE}}{{UNIT}} + 24px); height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'tabs_separator_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator,
					 {{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator:before, 
					 {{WRAPPER}} .rh-single-property-floor-plans-v2-tabs-separator:after' => 'background-color: {{VALUE}}',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'tabs_panels',
			[
				'label' => esc_html__( 'Panels', RHEA_TEXT_DOMAIN ),
				'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'tabs_panels_width',
			[
				'label'     => esc_html__( 'Width', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-content' => 'width: {{SIZE}}%;',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_panels_padding',
			[
				'label'     => esc_html__( 'Padding', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-panel-inner' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_panels_border_radius',
			[
				'label'     => esc_html__( 'Border Radius', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 200,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2' => '--border--radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tabs_panels_background',
			[
				'label'     => esc_html__( 'Background Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-panel' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Box_Shadow::get_type(),
			[
				'label'    => esc_html__( 'Box Shadow', RHEA_TEXT_DOMAIN ),
				'name'     => 'tabs_panels_box_shadow',
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-panel',
			]
		);

		$this->add_control(
			'panels_title',
			[
				'label'     => esc_html__( 'Title', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'panels_title_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-title' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'panels_title_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-title' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_title_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-title',
			]
		);

		$this->add_control(
			'panels_description',
			[
				'label'     => esc_html__( 'Description', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_responsive_control(
			'panels_description_margin',
			[
				'label'     => esc_html__( 'Margin', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::DIMENSIONS,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-content p' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'panels_description_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-content p' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_description_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-tab-content p',
			]
		);

		$this->add_control(
			'panels_price_heading',
			[
				'label'     => esc_html__( 'Price', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'panels_price_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-price' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_price_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-price',
			]
		);

		$this->add_control(
			'panels_price_postfix_heading',
			[
				'label'     => esc_html__( 'Price PostFix', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'panels_price_postfix_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-price .floor-price-post-fix' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_price_postfix_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-price .floor-price-post-fix',
			]
		);

		$this->add_control(
			'panels_meta_heading',
			[
				'label'     => esc_html__( 'Meta', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::HEADING,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'panels_meta_value_color',
			[
				'label'     => esc_html__( 'Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-meta-value' => 'color: {{VALUE}}',
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(),
			[
				'name'     => 'panels_meta_value_typography',
				'label'    => esc_html__( 'Typography', RHEA_TEXT_DOMAIN ),
				'global'   => [
					'default' => \Elementor\Core\Kits\Documents\Tabs\Global_Typography::TYPOGRAPHY_PRIMARY,
				],
				'selector' => '{{WRAPPER}} .rh-single-property-floor-plans-v2-meta-value',
			]
		);

		$this->add_control(
			'panels_meta_icon_light_color',
			[
				'label'     => esc_html__( 'Icon Light Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-meta-icon .rh-ultra-light' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_control(
			'panels_meta_icon_dark_color',
			[
				'label'     => esc_html__( 'Icon Dark Color', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-meta-icon .rh-ultra-dark' => 'fill: {{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'icon_size',
			[
				'label'     => esc_html__( 'Icon Size', RHEA_TEXT_DOMAIN ),
				'type'      => \Elementor\Controls_Manager::SLIDER,
				'range'     => [
					'px' => [
						'min' => 0,
						'max' => 128,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .rh-single-property-floor-plans-v2-meta-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render() {

		// Check if login required to vew property detail.
		if ( inspiry_prop_detail_login() ) {
			return;
		}

		$settings    = $this->get_settings_for_display();
		$widget_id   = $this->get_id();
		$floor_plans = get_post_meta( self::$property_id, 'inspiry_floor_plans', true );

		if ( ! empty( $floor_plans ) && is_array( $floor_plans ) && ! empty( $floor_plans[0]['inspiry_floor_plan_name'] ) ) {
			// Define the metadata to display.
			$meta_to_display = array(
				[
					'id'           => 'inspiry_floor_plan_bedrooms',
					'label'        => 'inspiry_bedrooms_field_label',
					'widget_label' => 'bedrooms_label',
					'default'      => esc_html__( 'Bedrooms', RHEA_TEXT_DOMAIN ),
					'icon'         => 'ultra-bedrooms',
					'post-fix'     => ''
				],
				[
					'id'           => 'inspiry_floor_plan_bathrooms',
					'label'        => 'inspiry_bathrooms_field_label',
					'widget_label' => 'bathrooms_label',
					'default'      => esc_html__( 'Bathrooms', RHEA_TEXT_DOMAIN ),
					'icon'         => 'ultra-bathrooms',
					'post-fix'     => ''
				],
				[
					'id'           => 'inspiry_floor_plan_size',
					'label'        => 'inspiry_area_field_label',
					'widget_label' => 'area_label',
					'default'      => esc_html__( 'Area', RHEA_TEXT_DOMAIN ),
					'icon'         => 'ultra-area',
					'post-fix'     => 'inspiry_floor_plan_size_postfix'
				],
			);
			?>
            <div id="rh-single-property-floor-plans-v2-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-floor-plans-v2 <?php realhomes_printable_section( 'floor-plans' ); ?>">
				<?php
				if ( ! empty( $settings['section_title'] ) ) {
					?>
                    <h4 class="rh-single-property-section-heading rh-single-property-section-heading-common"><?php echo esc_html( $settings['section_title'] ); ?></h4>
					<?php
				}
				?>
                <div id="rh-single-property-floor-plans-v2-tabs-<?php echo esc_attr( $widget_id ); ?>" class="rh-single-property-floor-plans-v2-tabs">
                    <ul class="rh-single-property-floor-plans-v2-tab-list">
						<?php
						$disable_tab_image = ( 'yes' === $settings['disable_tabs_image'] );
						foreach ( $floor_plans as $floor_plan ) {
							if ( ! empty( $floor_plan['inspiry_floor_plan_name'] ) ) {

								if ( $disable_tab_image ) {
									printf( '<li class="rh-single-property-floor-plans-v2-tab"><span class="rh-single-property-floor-plans-v2-tab-label">%s</span></li>', esc_html( $floor_plan['inspiry_floor_plan_name'] ) );
								} else {
									printf( '<li class="rh-single-property-floor-plans-v2-tab"><span class="rh-single-property-floor-plans-v2-tab-label">%s</span>%s</li>', esc_html( $floor_plan['inspiry_floor_plan_name'] ), $this->fp_image( $floor_plan, true ) );
								}
							}
						}
						?>
                    </ul>
                    <div class="rh-single-property-floor-plans-v2-tabs-separator"></div>
                    <div class="rh-single-property-floor-plans-v2-tab-content">
						<?php
						foreach ( $floor_plans as $floor_plan ) {
							?>
                            <div class="rh-single-property-floor-plans-v2-tab-panel">
								<?php echo $this->fp_image( $floor_plan ); ?>
                                <div class="rh-single-property-floor-plans-v2-tab-panel-inner">
									<?php
									if ( ! empty( $floor_plan['inspiry_floor_plan_name'] ) ) {
										?>
                                        <h4 class="rh-single-property-floor-plans-v2-title"><?php echo esc_html( $floor_plan['inspiry_floor_plan_name'] ); ?></h4>
										<?php
									}

									if ( ! empty( $floor_plan['inspiry_floor_plan_descr'] ) ) {
										?>
                                        <div class="rh-single-property-floor-plans-v2-desc"><p><?php echo esc_html( $floor_plan['inspiry_floor_plan_descr'] ); ?></p></div>
										<?php
									}
									?>
                                    <div class="rh-single-property-floor-plans-v2-meta-and-price">
                                        <div class="rh-single-property-floor-plans-v2-meta-wrapper">
											<?php
											foreach ( $meta_to_display as $value ) {
												if ( ! empty( $floor_plan[ $value['id'] ] ) ) {
													$post_fix = isset( $value['post-fix'] ) && ! empty( $value['post-fix'] ) ? $floor_plan[ $value['post-fix'] ] : '';
													?>
                                                    <div class="rh-single-property-floor-plans-v2-meta">
                                                        <div class="rh-single-property-floor-plans-v2-meta-icon">
                                                            <?php
                                                            rhea_property_meta_icon( $value['id'], $value['icon'] , true);
                                                            ?>
                                                        </div>
                                                        <span class="rh-single-property-floor-plans-v2-meta-value"><?php echo esc_html( $floor_plan[ $value['id'] ] . ' ' . $post_fix ); ?></span>
                                                    </div>
													<?php
												}
											}
											?>
                                        </div>
										<?php
										if ( ! empty( $floor_plan['inspiry_floor_plan_price'] ) ) {
											?>
                                            <span class="rh-single-property-floor-plans-v2-price"><?php echo ere_get_property_floor_price( $floor_plan ); ?></span>
											<?php
										}
										?>
                                    </div>
                                </div>
                            </div>
							<?php
						}
						?>
                    </div>
                </div>
            </div>
			<?php
		} else {
			rhea_print_no_result_for_editor();
		}
	}

	public function fp_image( $floor_plan, $is_tab_image = false ) {
		$output = '';

		if ( ! empty( $floor_plan['inspiry_floor_plan_image'] ) ) {
			$image = '<img src="' . esc_url( $floor_plan['inspiry_floor_plan_image'] ) . '" alt="' . esc_attr( $floor_plan['inspiry_floor_plan_name'] ) . '">';

			if ( $is_tab_image ) {
				$output = $image;
			} else {
				$output = '<a class="rh-single-property-floor-plans-v2-image" href="' . esc_url( $floor_plan['inspiry_floor_plan_image'] ) . '" data-fancybox="floor-plans">' . $image . '</a>';
			}
		}

		return $output;
	}

	private function initialize_property_id() {
		if ( rhea_is_preview_mode() && rhea_get_sample_property_id() ) {
			return rhea_get_sample_property_id();
		}

		return get_the_ID();
	}
}